<?php
session_start();
include "includes/header.php";

$pedidosFile = __DIR__."/../pedidos/pedidos.json";
$pedidos = file_exists($pedidosFile) ? json_decode(file_get_contents($pedidosFile), true) : [];

function getCapa($item){
    if (!empty($item['capa'])) return $item['capa'];
    if (!empty($item['relacionado']['capa'])) return $item['relacionado']['capa'];
    return "/imagens/default.png";
}

function statusInfo($status){
    $status = strtolower($status);
    switch($status){
        case "pendente": return ["Pedido Pendente","#facc15","fa-hourglass-half"];
        case "processando": return ["Em Andamento","#3b82f6","fa-spinner fa-spin"];
        case "concluido": return ["Disponível","#10b981","fa-check-circle"];
        case "cancelado": return ["Cancelado","#ef4444","fa-times-circle"];
    }
    return ["Desconhecido","#999","fa-question-circle"];
}
?>

<div class="page">

<h1 class="titulo">
    <i class="fa-solid fa-cart-shopping"></i> Gerenciar Pedidos
</h1>

<button id="verificarPedidosBtn" class="btn-full green">
    <i class="fa fa-check-circle"></i> Verificar Pedidos
</button>

<input type="text" id="busca" class="input" placeholder="🔍 Buscar pedido...">

<select id="statusFiltro" class="input">
    <option value="todos">Todos os Status</option>
    <option value="pendente">Pendente</option>
    <option value="processando">Processando</option>
    <option value="concluido">Disponível</option>
    <option value="cancelado">Cancelado</option>
</select>

<div id="listaPedidos" class="lista">

<?php foreach($pedidos as $item):
    $nome = $item['nome'] ?? $item['relacionado']['nome'] ?? 'Sem nome';
    $desc = $item['descricao'] ?? '';
    $status = $item['status'] ?? 'pendente';
    $capa = getCapa($item);
    [$st,$cor,$icon] = statusInfo($status);
?>

<div class="linha-item" data-status="<?= strtolower($status) ?>" data-nome="<?= strtolower($nome) ?>">

    <div class="linha-capa">
        <img src="<?= $capa ?>">
    </div>

    <div class="linha-info">
        <div class="linha-topo">
            <h3><?= htmlspecialchars($nome) ?></h3>
            <span class="badge" style="background:<?= $cor ?>">
                <i class="fa <?= $icon ?>"></i> <?= $st ?>
            </span>
        </div>

        <?php if($desc): ?>
        <p class="linha-desc"><?= htmlspecialchars($desc) ?></p>
        <?php endif; ?>
    </div>

    <div class="linha-acoes">
        <button class="btn green" onclick="alterarStatus('<?= $nome ?>','concluido','<?= $capa ?>','<?= $st ?>')">
            <i class="fa fa-check"></i>
        </button>
        <button class="btn blue" onclick="alterarStatus('<?= $nome ?>','processando','<?= $capa ?>','<?= $st ?>')">
            <i class="fa fa-spinner"></i>
        </button>
        <button class="btn gray" onclick="alterarStatus('<?= $nome ?>','cancelado','<?= $capa ?>','<?= $st ?>')">
            <i class="fa fa-ban"></i>
        </button>
        <button class="btn red" onclick="deletarPedido('<?= $nome ?>','<?= $capa ?>')">
            <i class="fa fa-trash"></i>
        </button>
    </div>
</div>

<?php endforeach; ?>
</div>
</div>

<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
const busca = document.getElementById("busca");
const statusFiltro = document.getElementById("statusFiltro");

function filtrar(){
    document.querySelectorAll(".linha-item").forEach(i=>{
        const nome = i.dataset.nome;
        const status = i.dataset.status;
        const okNome = nome.includes(busca.value.toLowerCase());
        const okStatus = statusFiltro.value==="todos" || status===statusFiltro.value;
        i.style.display = (okNome && okStatus) ? "flex":"none";
    });
}

busca.addEventListener("input", filtrar);
statusFiltro.addEventListener("change", filtrar);

function alterarStatus(nome, novoStatus, capa, statusAtual){
    Swal.fire({
        title: "Alterar status?",
        html: `
            <div style="display:flex;gap:10px;align-items:center">
                <img src="${capa}" style="width:70px;height:90px;border-radius:10px;background:#000;object-fit:contain">
                <div style="text-align:left">
                    <b>${nome}</b><br>
                    <small>De: ${statusAtual}</small><br>
                    <small>Para: ${novoStatus}</small>
                </div>
            </div>
        `,
        icon: "question",
        showCancelButton: true,
        confirmButtonText: "Sim",
        cancelButtonText: "Cancelar",
        backdrop: 'rgba(0,0,0,1)'
    }).then(res=>{
        if(res.isConfirmed){
            fetch("alterar_status.php",{
                method:"POST",
                headers:{'Content-Type':'application/json'},
                body: JSON.stringify({nome, novoStatus})
            }).then(()=>location.reload());
        }
    });
}

function deletarPedido(nome, capa){
    Swal.fire({
        title:"Excluir pedido?",
        html: `
            <div style="display:flex;gap:10px;align-items:center">
                <img src="${capa}" style="width:70px;height:90px;border-radius:10px;background:#000;object-fit:contain">
                <div><b>${nome}</b><br><small>Essa ação não pode ser desfeita</small></div>
            </div>
        `,
        icon:"warning",
        showCancelButton:true,
        confirmButtonText:"Excluir",
        cancelButtonText:"Cancelar",
        backdrop:'rgba(0,0,0,1)'
    }).then(res=>{
        if(res.isConfirmed){
            fetch("delete_pedido.php",{
                method:"POST",
                headers:{'Content-Type':'application/json'},
                body: JSON.stringify({nome})
            }).then(()=>location.reload());
        }
    });
}

document.getElementById("verificarPedidosBtn").addEventListener("click", ()=>{
    Swal.fire({
        title:"Verificando pedidos...",
        html:"<i class='fa fa-spinner fa-spin' style='font-size:40px'></i>",
        showConfirmButton:false,
        backdrop:'rgba(0,0,0,1)'
    });

    fetch("verificar_pedidos.php")
    .then(r=>r.json())
    .then(()=>{
        Swal.fire({
            title:"Pedidos verificados!",
            icon:"success",
            confirmButtonText:"Ok",
            backdrop:'rgba(0,0,0,1)'
        }).then(()=>location.reload());
    });
});
</script>

<style>
.page{max-width:1100px;margin:auto;padding:90px 16px;color:var(--text-primary)}
.titulo{text-align:center;margin-bottom:20px;font-size:26px}

.input{width:100%;padding:14px;border-radius:14px;margin-bottom:12px;border:2px solid var(--border-color);background:var(--menu-secondary);color:var(--text-primary)}

.btn-full{width:100%;padding:16px;border-radius:16px;font-weight:bold;border:none;margin-bottom:14px;cursor:pointer}
.btn-full.green{background:#10b981;color:#fff}

.lista{display:flex;flex-direction:column;gap:10px}

.linha-item{
    display:flex;
    gap:12px;
    align-items:center;
    background:var(--menu-secondary);
    border:1.5px solid var(--border-color);
    border-radius:16px;
    padding:12px;
}

.linha-capa img{
    width:60px;
    height:80px;
    border-radius:8px;
    background:#000;
    object-fit:contain;
}

.linha-info{flex:1;display:flex;flex-direction:column;gap:4px}

.linha-topo{display:flex;justify-content:space-between;gap:10px}
.linha-topo h3{font-size:15px;margin:0}

.linha-desc{font-size:13px;opacity:.85}

.badge{font-size:12px;padding:4px 10px;border-radius:999px;color:#fff;white-space:nowrap}

.linha-acoes{display:flex;gap:6px}

.btn{
    width:36px;
    height:36px;
    border-radius:10px;
    border:none;
    display:flex;
    align-items:center;
    justify-content:center;
    color:#fff;
}

.green{background:#10b981}
.blue{background:#3b82f6}
.gray{background:#6b7280}
.red{background:#ef4444}

@media(max-width:768px){
    .linha-item{flex-direction:column;align-items:stretch}
    .linha-capa img{width:100%;height:160px}
    .linha-acoes{display:grid;grid-template-columns:repeat(4,1fr);gap:8px}
    .btn{width:100%;height:42px;font-size:16px}
}
</style>